<?php

namespace App\Http\Controllers;
use App\Models\PlanEstudio;
use Illuminate\Http\Request;
use App\Models\Periodo;
use App\Models\Semestre;
use App\Models\Curso;
use Illuminate\Support\Facades\DB;
use App\Models\Horario;
use App\Models\Docente;
use App\Models\CursoDocente;
use Illuminate\Support\Facades\Validator;
use App\Models\Turno;
use App\Models\Seccion;
use App\Models\Pabellon;
use App\Models\Aula;
use Illuminate\Support\Facades\Auth;
class HorarioController extends Controller
{
    /**
     * Display a listing of the resource.
     */
   public function index()
{
    $periodoActivo = Periodo::where('activo', 1)->first();

    // Lista de pabellones para el select
    $pabellones = Pabellon::orderBy('nombre')->get();

    // Aulas vacías al inicio (se llenarán por AJAX)
    $aulas = [];

    $horarios = [];

    if ($periodoActivo) {
        $horarios = Horario::with(['docente', 'curso', 'periodo', 'planEstudio', 'aula.pabellon'])
                    ->where('id_periodo', $periodoActivo->id)
                    ->get();
    }

    return view('horarios.index', compact('horarios', 'periodoActivo', 'pabellones', 'aulas'));
}


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $docentes = Docente::all();
        $turnos = Turno::all();
        $secciones = Seccion::all();
        $cursos = Curso::all();
        $planesEstudio = PlanEstudio::where('activo', true)->get();
        $periodos = Periodo::where('activo', 1)->orderBy('id', 'desc')->get();
        // Lista de pabellones para el select
    $pabellones = Pabellon::orderBy('nombre')->get();

        return view('horarios.create', compact('docentes', 'cursos', 'planesEstudio', 'periodos','turnos','secciones', 'pabellones'));
    }



    /**
     * Store a newly created resource in storage.
     */
    public function store1(Request $request)
{
    $validator = Validator::make($request->all(), [
        'id_periodo'      => 'required|exists:periodos,id',
        'id_docente'      => 'required|exists:docentes,id',
        'id_plan_estudio' => 'required|exists:planes_estudio,id',
        'id_curso'        => 'required|exists:cursos,id',
        'hora_inicio'     => 'required|date_format:H:i',
        'hora_fin'        => 'required|date_format:H:i|after:hora_inicio',
        'dia'             => 'required|in:Lunes,Martes,Miércoles,Jueves,Viernes,Sábado,Domingo',

        // NUEVO CAMPO DE AULA
        'id_aula'         => 'required|exists:aulas,id',

        'tipo_aula'       => 'required|in:Aula,Laboratorio',
        'numero_aula'     => 'required|string|max:10',
        'frecuencia'      => 'required|in:Semanal,Quincenal,Mensual',
        'capacidad'       => 'required|integer|min:1',
        'id_turno'        => 'required|exists:turnos,id',
        'id_seccion'      => 'required|exists:secciones,id',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'errors' => $validator->errors(),
        ], 422);
    }

    // Validar conflicto cruzado
    $existeConflicto = Horario::where('id_docente', $request->id_docente)
        ->where('dia', $request->dia)
        ->where('id_turno', $request->id_turno)
        ->where('id_periodo', $request->id_periodo)
        ->where(function ($query) use ($request) {
            $query->whereBetween('hora_inicio', [$request->hora_inicio, $request->hora_fin])
                ->orWhereBetween('hora_fin', [$request->hora_inicio, $request->hora_fin])
                ->orWhere(function ($q) use ($request) {
                    $q->where('hora_inicio', '<', $request->hora_inicio)
                      ->where('hora_fin', '>', $request->hora_fin);
                });
        })
        ->exists();

    if ($existeConflicto) {
        return response()->json([
            'success' => false,
            'message' => 'Conflicto: el docente ya tiene un horario en ese rango.',
        ], 409);
    }

    // Relación curso-docente
    CursoDocente::firstOrCreate([
        'curso_id'   => $request->id_curso,
        'docente_id' => $request->id_docente,
    ]);

    // Crear horario
    $horario = Horario::create([
        'id_docente'      => $request->id_docente,
        'id_curso'        => $request->id_curso,
        'id_periodo'      => $request->id_periodo,
        'id_plan_estudio' => $request->id_plan_estudio,
        'hora_inicio'     => $request->hora_inicio,
        'hora_fin'        => $request->hora_fin,
        'dia'             => $request->dia,
        'id_aula'         => $request->id_aula, // <<--- NUEVO
        'tipo_aula'       => $request->tipo_aula,
        'numero_aula'     => $request->numero_aula,
        'frecuencia'      => $request->frecuencia,
        'capacidad'       => $request->capacidad,
        'id_turno'        => $request->id_turno,
        'id_seccion'      => $request->id_seccion,
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Horario registrado correctamente.',
        'horario' => $horario,
    ]);
}

public function store(Request $request, \App\Services\MoodleService $moodle)
{
    $validator = Validator::make($request->all(), [
        'id_periodo'      => 'required|exists:periodos,id',
        'id_docente'      => 'required|exists:docentes,id',
        'id_plan_estudio' => 'required|exists:planes_estudio,id',
        'id_curso'        => 'required|exists:cursos,id',
        'hora_inicio'     => 'required|date_format:H:i',
        'hora_fin'        => 'required|date_format:H:i|after:hora_inicio',
        'dia'             => 'required|in:Lunes,Martes,Miércoles,Jueves,Viernes,Sábado,Domingo',
        'id_aula'         => 'required|exists:aulas,id',
        'tipo_aula'       => 'required|in:Aula,Laboratorio',
        'numero_aula'     => 'required|string|max:10',
        'frecuencia'      => 'required|in:Semanal,Quincenal,Mensual',
        'capacidad'       => 'required|integer|min:1',
        'id_turno'        => 'required|exists:turnos,id',
        'id_seccion'      => 'required|exists:secciones,id',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'errors' => $validator->errors(),
        ], 422);
    }

    // Validar conflicto de horarios
    $existeConflicto = Horario::where('id_docente', $request->id_docente)
        ->where('dia', $request->dia)
        ->where('id_turno', $request->id_turno)
        ->where('id_periodo', $request->id_periodo)
        ->where(function ($query) use ($request) {
            $query->whereBetween('hora_inicio', [$request->hora_inicio, $request->hora_fin])
                ->orWhereBetween('hora_fin', [$request->hora_inicio, $request->hora_fin])
                ->orWhere(function ($q) use ($request) {
                    $q->where('hora_inicio', '<', $request->hora_inicio)
                      ->where('hora_fin', '>', $request->hora_fin);
                });
        })
        ->exists();

    if ($existeConflicto) {
        return response()->json([
            'success' => false,
            'message' => 'Conflicto: el docente ya tiene un horario en ese rango.',
        ], 409);
    }

    // Relación curso-docente
    $cursoDocente = CursoDocente::firstOrCreate([
        'curso_id'   => $request->id_curso,
        'docente_id' => $request->id_docente,
    ]);

    // Crear horario
    $horario = Horario::create([
        'id_docente'      => $request->id_docente,
        'id_curso'        => $request->id_curso,
        'id_periodo'      => $request->id_periodo,
        'id_plan_estudio' => $request->id_plan_estudio,
        'hora_inicio'     => $request->hora_inicio,
        'hora_fin'        => $request->hora_fin,
        'dia'             => $request->dia,
        'id_aula'         => $request->id_aula,
        'tipo_aula'       => $request->tipo_aula,
        'numero_aula'     => $request->numero_aula,
        'frecuencia'      => $request->frecuencia,
        'capacidad'       => $request->capacidad,
        'id_turno'        => $request->id_turno,
        'id_seccion'      => $request->id_seccion,
    ]);

    // ===============================================
    //   ASIGNAR PROFESOR AL CURSO EN MOODLE
    // ===============================================

    $docente = Docente::find($request->id_docente);
    $usuario = $docente->usuario;

    if ($usuario->moodle_user_id) {
        $curso = Curso::find($request->id_curso);

        if ($curso->moodle_course_id) {
            $moodle->asignarProfesor(
                $usuario->moodle_user_id,    // moodle user id
                $curso->moodle_course_id     // moodle course id
            );
        }
    }

    return response()->json([
        'success' => true,
        'message' => 'Horario registrado correctamente y profesor asignado en Moodle.',
        'horario' => $horario,
    ]);
}




    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $horario = Horario::findOrFail($id);
        return response()->json($horario);  // Retorna los datos del horario
    }
    
    public function update(Request $request, $id)
    {
        $horario = Horario::findOrFail($id);
        $horario->update($request->all());
    
        return response()->json($horario);  // Retorna el horario actualizado
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, $id)
    {
        $horario = Horario::findOrFail($id);

        // Obtener los IDs del curso, docente y periodo asociados al horario
        $cursoId = $horario->id_curso;
        $docenteId = $horario->id_docente;
        $periodoId = $horario->id_periodo;

        // Eliminar el horario
        $horario->delete();

        // Verificar si el docente sigue dictando el curso en otros periodos
        $existenOtrosHorarios = Horario::where('id_curso', $cursoId)
                                    ->where('id_docente', $docenteId)
                                    ->where('id_periodo', '!=', $periodoId)
                                    ->exists();

        // Si no existen más horarios para este curso y docente en otros periodos, eliminamos la relación
        if (!$existenOtrosHorarios) {
            // Eliminar la relación entre el curso y el docente si ya no existen otros horarios para el docente en ese curso
            CursoDocente::where('curso_id', $cursoId)
                        ->where('docente_id', $docenteId)
                        ->delete();
        }

        // Devolvemos una respuesta JSON para AJAX
        return response()->json(['success' => true]);
    }


    public function planesPorPeriodo($periodoId)
    {
        $planes = PlanEstudio::whereHas('periodos', function ($q) use ($periodoId) {
            $q->where('periodo_id', $periodoId);
        })->get();
    
        return response()->json($planes);
    }
    

    public function cursosPorPlan($planId)
    {
        // Si estás usando una tabla intermedia llamada cursos_plan_estudio
        $cursos = Curso::whereHas('planesEstudio', function ($q) use ($planId) {
            $q->where('plan_estudio_id', $planId);
        })->get();

        return response()->json($cursos);
    }

    // En tu controlador HorarioController
    public function obtenerDocentes()
{
    $docentes = Docente::select('id', 'nombre', 'apellido')
        ->orderBy('apellido')
        ->orderBy('nombre')
        ->get();

    // Devuelve un array con el ID y el nombre completo
    return response()->json($docentes->map(function ($docente) {
        return [
            'id' => $docente->id,
            'nombre_completo' => $docente->nombre_completo, // Usa el accesor
        ];
    }));
}
    public function porDocentePeriodo($docenteId, $periodoId)
    {
        $horarios = Horario::with('curso')
            ->where('id_docente', $docenteId)
            ->where('id_periodo', $periodoId)
            ->get();

        return response()->json($horarios);
    }


    public function verHorarioDocente()
    {
        // Obtener el id del docente autenticado
        $docente_id = Auth::user()->docente->id ?? null;

        if (!$docente_id) {
            return redirect()->route('home')->with('error', 'No se encuentra autenticado como docente.');
        }

        // Obtener los horarios del docente (relación con la tabla 'horarios')
        $horarios = Horario::where('id_docente', $docente_id)
            ->with(['curso', 'seccion'])  // Cargar la relación con los cursos y secciones
            ->orderBy('hora_inicio') // Ordenar por hora de inicio
            ->get();

        // Devolver a la vista con los horarios
        return view('horarios.docente_horario', compact('horarios'));

    }

public function aulasPorPabellon($id)
{
    $aulas = Aula::where('id_pabellon', $id)->orderBy('nombre')->get();

    return response()->json([
        'success' => true,
        'aulas' => $aulas
    ]);
}

}
