<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    /**
     * Mostrar la lista de roles.
     */
    public function index()
    {
        $roles = Role::all(); // Obtiene todos los roles
        
        return view('roles.index', compact('roles'));
    }

    /**
     * Almacenar un nuevo rol.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:50|unique:roles,nombre',
            'descripcion' => 'nullable|string',
        ]);

        try {
            $role = Role::create($request->only('nombre', 'descripcion'));
            return response()->json(['success' => 'Rol creado correctamente.', 'role' => $role]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al crear el rol.'], 500);
        }
    }

    /**
     * Mostrar los detalles de un rol específico.
     */
    public function show($id)
    {
        $role = Role::find($id);

        if (!$role) {
            return response()->json(['error' => 'Rol no encontrado.'], 404);
        }

        return response()->json($role);
    }

    /**
     * Actualizar un rol existente.
     */
    public function update(Request $request, $id)
    {
        $role = Role::find($id);

        if (!$role) {
            return response()->json(['error' => 'Rol no encontrado.'], 404);
        }

        $request->validate([
            'nombre' => 'required|string|max:50|unique:roles,nombre,' . $id,
            'descripcion' => 'nullable|string',
        ]);

        try {
            $role->update($request->only('nombre', 'descripcion'));
            return response()->json(['success' => 'Rol actualizado correctamente.', 'role' => $role]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al actualizar el rol.'], 500);
        }
    }

    /**
     * Eliminar un rol específico.
     */
    public function destroy($id)
    {
        $role = Role::find($id);

        if (!$role) {
            return response()->json(['error' => 'Rol no encontrado.'], 404);
        }

        try {
            $role->delete();
            return response()->json(['success' => 'Rol eliminado correctamente.']);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al eliminar el rol.'], 500);
        }
    }

    public function cambiarEstado(Request $request, $id)
    {
        $role = Role::findOrFail($id);
        $role->estado = $request->estado;
        $role->save();

        return response()->json(['message' => 'Estado actualizado correctamente.']);
    }

}
